<?php
/**
 * Plugin Name: WooCommerce Purolator
 * Plugin URI: https://woocommerce.com/products/purolator-shipping-method/
 * Description: Automatic Shipping Calculation using the Purolator Shipping API for WooCommerce
 * Version: 2.4.17
 * Author: WooCommerce
 * Author URI: https://woocommerce.com/
 * Tested up to: 5.1
 * WC requires at least: 2.6
 * WC tested up to: 3.5
 * Copyright: © 2009-2017 WooCommerce.
 * License: GNU General Public License v3.0
 * License URI: http://www.gnu.org/licenses/gpl-3.0.html
 *
 * Woo: 18730:64d6d3f310029beecd8b864742f219bf
 */

/**
 * Required functions
 */
if ( ! function_exists( 'woothemes_queue_update' ) ) {
	require_once( 'woo-includes/woo-functions.php' );
}

/**
 * Plugin updates
 */
woothemes_queue_update( plugin_basename( __FILE__ ), '64d6d3f310029beecd8b864742f219bf', '18730' );

define( 'WC_SHIPPING_PUROLATOR_VERSION', '2.4.16' );

/**
 * Check if WooCommerce is active
 */
if ( is_woocommerce_active() ) {
	class WC_Purolator {
		/**
		 * Constructor
		 */
		public function __construct() {
			add_action( 'admin_init', array( $this, 'maybe_install' ), 5 );
			add_action( 'admin_init', array( $this, 'load_order_functions' ) );
			add_action( 'init', array( $this, 'load_plugin_textdomain' ) );
			add_action( 'init', array( $this, 'load_ajax_processes' ) );
			add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), array( $this, 'plugin_action_links' ) );
			add_action( 'woocommerce_shipping_init', array( $this, 'init' ) );
			add_filter( 'woocommerce_shipping_methods', array( $this, 'add_method' ) );
			add_action( 'admin_notices', array( $this, 'upgrade_notice' ) );
			add_action( 'wp_ajax_purolator_dismiss_upgrade_notice', array( $this, 'dismiss_upgrade_notice' ) );
		}

		/**
		 * Localisation
		 */
		public function load_plugin_textdomain() {
			load_plugin_textdomain( 'woocommerce-shipping-purolator', false, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );
		}

		/**
		 * Plugin page links
		 */
		public function plugin_action_links( $links ) {
			$plugin_links = array(
				'<a href="' . admin_url( 'admin.php?page=wc-settings&tab=shipping&section=purolator' ) . '">' . __( 'Settings', 'woocommerce-shipping-purolator' ) . '</a>',
				'<a href="http://support.woocommes.com/">' . __( 'Support', 'woocommerce-shipping-purolator' ) . '</a>',
				'<a href="http://docs.woocommerce.com/document/purolator-shipping/">' . __( 'Docs', 'woocommerce-shipping-purolator' ) . '</a>',
			);
			return array_merge( $plugin_links, $links );
		}

		/**
		 * Load gateway class
		 */
		public function init() {
			if ( version_compare( WC_VERSION, '2.6.0', '<' ) ) {
				include_once( dirname( __FILE__ ) . '/includes/class-wc-shipping-purolator-deprecated.php' );
			} else {
				include_once( dirname( __FILE__ ) . '/includes/class-wc-shipping-purolator.php' );
			}
			include_once( dirname( __FILE__ ) . '/includes/class-wc-shipping-purolator-privacy.php' );
		}

		public function load_order_functions() {
			if ( version_compare( WC_VERSION, '2.6.0', '<' ) ) {
				return;
			}

			require_once( dirname( __FILE__ ) . '/includes/class-wc-shipping-purolator-order.php' );
		}

		/**
		 * Loads all ajax processes.
		 *
		 * @since 2.4.0
		 * @version 2.4.0
		 */
		public function load_ajax_processes() {
			require_once( dirname( __FILE__ ) . '/includes/class-wc-shipping-purolator-ajax.php' );
		}

		/**
		 * Add method to WC
		 */
		public function add_method( $methods ) {
			if ( version_compare( WC_VERSION, '2.6.0', '<' ) ) {
				$methods[] = 'WC_Shipping_Purolator';
			} else {
				$methods['purolator'] = 'WC_Shipping_Purolator';
			}

			return $methods;
		}

		/**
		 * Checks the plugin version
		 *
		 * @access public
		 * @since 2.4.0
		 * @version 2.4.0
		 * @return bool
		 */
		public function maybe_install() {
			// only need to do this for versions less than 2.4.0 to migrate
			// settings to shipping zone instance
			$doing_ajax = defined( 'DOING_AJAX' ) && DOING_AJAX;
			if ( ! $doing_ajax
			     && ! defined( 'IFRAME_REQUEST' )
			     && version_compare( WC_VERSION, '2.6.0', '>=' )
			     && version_compare( get_option( 'wc_purolator_version' ), '2.4.0', '<' ) ) {

				$this->install();

			}

			return true;
		}

		/**
		 * Update/migration script
		 *
		 * @since 2.4.0
		 * @version 2.4.0
		 * @access public
		 * @return bool
		 */
		public function install() {
			// get all saved settings and cache it
			$purolator_settings = get_option( 'woocommerce_purolator_settings', false );

			// settings exists
			if ( $purolator_settings ) {
				global $wpdb;

				// unset un-needed settings
				unset( $purolator_settings['enabled'] );
				unset( $purolator_settings['availability'] );
				unset( $purolator_settings['countries'] );

				// first add it to the "rest of the world" zone when no purolator
				// instance.
				if ( ! $this->is_zone_has_purolator( 0 ) ) {
					$wpdb->query( $wpdb->prepare( "INSERT INTO {$wpdb->prefix}woocommerce_shipping_zone_methods ( zone_id, method_id, method_order, is_enabled ) VALUES ( %d, %s, %d, %d )", 0, 'purolator', 1, 1 ) );
					// add settings to the newly created instance to options table
					$instance = $wpdb->insert_id;
					add_option( 'woocommerce_purolator_' . $instance . '_settings', $purolator_settings );
				}

				update_option( 'woocommerce_purolator_show_upgrade_notice', 'yes' );
			}

			update_option( 'wc_purolator_version', WC_SHIPPING_PUROLATOR_VERSION );
		}

		/**
		 * Show the user a notice for plugin updates
		 *
		 * @since 2.4.0
		 */
		public function upgrade_notice() {
			$show_notice = get_option( 'woocommerce_purolator_show_upgrade_notice' );

			if ( 'yes' !== $show_notice ) {
				return;
			}

			$query_args = array( 'page' => 'wc-settings', 'tab' => 'shipping' );
			$zones_admin_url = add_query_arg( $query_args, get_admin_url() . 'admin.php' );
			?>
			<div class="notice notice-success is-dismissible wc-purolator-notice">
				<p><?php echo sprintf( __( 'Purolator now supports shipping zones. The zone settings were added to a new Purolator method on the "Rest of the World" Zone. See the zones %1$shere%2$s ', 'woocommerce-shipping-purolator' ), '<a href="' . $zones_admin_url . '">','</a>' ); ?></p>
			</div>

			<script type="application/javascript">
				jQuery( '.notice.wc-purolator-notice' ).on( 'click', '.notice-dismiss', function () {
					var data = {
						action: 'purolator_dismiss_upgrade_notice'
					};

					jQuery.post( '<?php echo admin_url( 'admin-ajax.php' ); ?>', data );
				});
			</script>
			<?php
		}

		/**
		 * Turn of the dismisable upgrade notice.
		 * @since 2.4.0
		 */
		public function dismiss_upgrade_notice() {
			update_option( 'woocommerce_purolator_show_upgrade_notice', 'no' );
		}

		/**
		 * Helper method to check whether given zone_id has purolator method instance.
		 *
		 * @since 2.4.0
		 *
		 * @param int $zone_id Zone ID
		 *
		 * @return bool True if given zone_id has purolator method instance
		 */
		public function is_zone_has_purolator( $zone_id ) {
			global $wpdb;
			return (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(instance_id) FROM {$wpdb->prefix}woocommerce_shipping_zone_methods WHERE method_id = 'purolator' AND zone_id = %d", $zone_id ) ) > 0;
		}
	}

	new WC_Purolator();
}
